/*-----------------------------------------------------------------------------------------------------------------------------
FileName: particleDetector.c 
Autors: Mathieu Gisselbrecht
Date : Creation -  Avril 2006
-----------------------------------------------------------------------------------------------------------------------------*/
//-----------------------------------------------------------------------------------------------------------------------------
// All structures are 2-byte-aligned.
#if TARGET_CPU_PPC
	#pragma options align=mac68k
#endif
#ifdef _WINDOWS_
	#pragma pack(2)
#endif

//-----------------------------------------------------------------------------------------------------------------------------
// PROJECT INCLUDE 
//-----------------------------------------------------------------------------------------------------------------------------
#include "igorTools.h"
#include "particleDetector.h"
#include "XYDetector.h"



//-----------------------------------------------------------------------------------------------------------------------------
// MISC #DEFINEs
//-----------------------------------------------------------------------------------------------------------------------------
#pragma mark Defines


//-----------------------------------------------------------------------------------------------------------------------------
// LOCAL VARIABLES
//-----------------------------------------------------------------------------------------------------------------------------
#pragma mark Local Variables 


//-----------------------------------------------------------------------------------------------------------------------------
// LOCAL PROTOTYPES 
//-----------------------------------------------------------------------------------------------------------------------------
#pragma mark Local Prototypes 

static short sigbit(int mask);
static short moveigorFolder(DataFolderHandle *originalFolder);
static short moveigorDetectorFolder(DataFolderHandle *originalFolder, int num);	
static short setigorDetector(TDetector *Detector);

//-----------------------------------------------------------------------------------------------------------------------------
//  									  				COMMON CODE 
//-----------------------------------------------------------------------------------------------------------------------------
#pragma mark ----------------------------
//-----------------------------------------------------------------------------------------------------------------------------
static short sigbit(int mask)
{ 
	int i=0, ok=mask&1;
	
	while ((!ok)&&(i<32)) {
		mask=mask>>1;
		ok=mask&1;
		i++;
	};
	
	return i;	
}
//-----------------------------------------------------------------------------------------------------------------------------
static short moveigorFolder(DataFolderHandle *originalFolder)
{
	int error=0;
			
	DataFolderHandle  rootFolder, varFolder, detectorListFolder;
	if (error = GetCurrentDataFolder(originalFolder)) goto end;	 	
		// Check Data folder  	
  	if (error = GetRootDataFolder(0, &rootFolder)) goto end;
	if (error = GetNamedDataFolder(NULL, PATHVAR, &varFolder)) {
	  	if (error = NewDataFolder(rootFolder, "var", &varFolder)) goto end;
		}
	if (error = GetNamedDataFolder(NULL, PATHDET, &detectorListFolder)) {
	  	if (error = NewDataFolder(varFolder, "detector", &detectorListFolder)) goto end;
		}		
	if (error = SetCurrentDataFolder(detectorListFolder)) goto end;
	
end:	
	return error;	
}
//-----------------------------------------------------------------------------------------------------------------------------
static short moveigorDetectorFolder(DataFolderHandle *originalFolder, int num)	
{
	DataFolderHandle  detectorListFolder, detectorFolder;

	char name[255];
	short error=0;
	
	if (error=moveigorFolder(originalFolder)) goto end;
	if (error = GetCurrentDataFolder(&detectorListFolder)) goto end;	 	

	sprintf(name,"%i",num);
	if (error = GetNamedDataFolder(detectorListFolder, name, &detectorFolder)) {
	  	if (error = NewDataFolder(detectorListFolder, name, &detectorFolder)) goto end;
		} 
	if (error = SetCurrentDataFolder(detectorFolder)) goto end;

end:	
	return error;
	
}
//-----------------------------------------------------------------------------------------------------------------------------
#pragma mark ----------------------------
//-----------------------------------------------------------------------------------------------------------------------------

void newEvent(TDetector *Detector)
{

	switch (Detector->type) {
			case 0x0F: // FOR (X,Y,T) 3D LINE DETECTOR :  Reset the number of hits to zero 
					InitXYEvent(Detector);
					
				break;
			case 0x17: // Other Detector type (X,Y,T) with delay lines
				break; 						
		}
}
//-----------------------------------------------------------------------------------------------------------------------------

short addEvent(TDetector *Detector, long tdcTime, long tdcChannel)
{	
	long type;
	
	type=Detector->type;		
	switch (type) {
		case 0x0F:		
				addXYEvent(Detector,tdcTime,tdcChannel);
			break;
		case 0x17: // Other Detector type (X,Y,T) with delay lines
			break; 	
	}
	return 0;
}
//-----------------------------------------------------------------------------------------------------------------------------
short correctEvent(TListDetector *ListOfDetector)		
{
	TDetector *Detector;
	int numDetector, type;
	short error=0;
	
	for (numDetector=0;numDetector<ListOfDetector->nbDetector;numDetector++) {
		Detector=&ListOfDetector->Detector[numDetector];
		type=Detector->type;
		switch (type) {
			case 0x0F:		
				if (error=correctXYEvent(Detector)) return error;
			break;
		case 0x17: // Other Detector type (X,Y,T) with delay lines
			break;
		}
	}
	return 0;	
}


//-----------------------------------------------------------------------------------------------------------------------------
short analyzeEvent(TListDetector *ListOfDetector)
{	
	TDetector *Detector;
	int mode, type, minOrder, maxOrder, coincidenceOrder;
	int i, result=0; 

	mode=ListOfDetector->modeDetector;
	for (i=0;i<ListOfDetector->nbDetector;i++) {
		Detector =&ListOfDetector->Detector[i];
		minOrder = Detector-> minOrder;
		maxOrder = Detector-> maxOrder;
		type = Detector-> type;	
		switch (type) {
			case 0x0F: // FOR (X,Y,T) 3D LINE DETECTOR :  Reset the number of hits to zero  
					coincidenceOrder=localizedXYEvent(Detector); // return -> -1 no hit at all, -> 0 no localization, -> coincidence order
				break;
			case 0x17: // Other Detector type (X,Y,T) with delay lines
				break;
		}
		if(coincidenceOrder>=0) {
				Detector->stat[coincidenceOrder]+=1; 	// Make the static on the coincidence
		}
		if((coincidenceOrder>maxOrder)||(coincidenceOrder<minOrder)) result=-1;	// Drop event if the coincidence order is not good

	}
	return result;
}
#pragma mark ----------------------------

//-----------------------------------------------------------------------------------------------------------------------------
short createDetector(TDetector *Detector, int num, char *name, int type, 
			int minOrder, int maxOrder, long tof_window, long tof_offset, long tof_bin, long image_bin)
{
	int error=0;	
	
	// Generic
	strcpy(Detector->name,name);
	strncpy(Detector->suffix,name,1);
	Detector->num=num;
	Detector->type=type;
	Detector->minOrder=minOrder;
	Detector->maxOrder=maxOrder;
	
	Detector->tof_window= tof_window;
	Detector->tof_offset= tof_offset;
	Detector->tof_bin	= tof_bin;
	Detector->image_bin	= image_bin;	
	return error;	
}


//-----------------------------------------------------------------------------------------------------------------------------
short updateDetector(TDetector *Detector, int mode)
{
	DataFolderHandle originalFolder;
	waveHndl	w;	

	DOUBLE 		val1, val2;	
	
	long 		numRows=0, numColumns=0;
	int 		error=0, flags = OVERWRITE,i ;
	char 		s[100];
		
	// Global information from the PSD
	if (error = moveigorDetectorFolder(&originalFolder,Detector->num)) goto end;;
	switch (mode) {
		case CREATE:
				MemClear(Detector->stat, 	sizeof(Detector->stat));
				MemClear(Detector->tof, 	sizeof(Detector->tof));
				MemClear(Detector->image, 	sizeof(Detector-> image));
				MemClear(Detector->projX, 	sizeof(Detector->projX));
				MemClear(Detector->projY, 	sizeof(Detector->projY));
				// Create and clear Igor wave
				if (error=MakeAWave("stat", flags, &w, MAXCOINCIDENCEORDER, 1, NT_I32)) goto end;
				if (error=MakeAWave("all_T", flags, &w, Detector->tof_window, 1, NT_I32)) goto end;
				if (error=MakeAWave("all_X", flags, &w, Detector->image_bin, 1, NT_I32)) goto end;
				if (error=MakeAWave("all_Y", flags, &w, Detector->image_bin, 1, NT_I32)) goto end;
				if (error=MakeAWave("image", flags, &w, Detector->image_bin, Detector->image_bin, NT_I32)) goto end;								
				sprintf(s,"stat = 0; all_T = 0; all_X = 0; all_Y = 0; image =0");
				if (error=XOPSilentCommand(s)) goto end;
				Detector->efficiency=0;
				if (error = SetIgorFloatingVar("efficiency",&Detector->efficiency,1)) goto end;				
				break;

		case ADD:
				numRows=MAXCOINCIDENCEORDER;
				if (error=CheckWave("stat", &w, &numRows, &numColumns)) goto end;
				numRows=Detector->tof_window;
				if (error=CheckWave("all_T", &w, &numRows, &numColumns)) goto end;
				numRows=Detector->image_bin;
				if (error=CheckWave("all_X", &w, &numRows, &numColumns)) goto end;
				numRows=Detector->image_bin;
				if (error=CheckWave("all_Y", &w, &numRows, &numColumns)) goto end;
				numRows=Detector->image_bin;
				numColumns=Detector->image_bin;
				if (error=CheckWave("image", &w, &numRows, &numColumns)) goto end;
     			if (FetchNumVar("efficiency",&val1,&val2)==-1) { 
  						if (error = SetIgorFloatingVar("efficiency",&Detector->efficiency,1)) goto end;
  					}
  				else Detector->efficiency=(long)val1;				
			break;

		case END_OF_ACQUISITION:
				if (error=AddTable2IgorWave("stat", Detector->stat, MAXCOINCIDENCEORDER,1)) goto end;
				if (error=AddTable2IgorWave("all_T", Detector->tof, Detector->tof_window, 1)) goto end;		
				if (error=AddTable2IgorWave("all_X", Detector->projX, Detector->image_bin, 1)) goto end;
				if (error=AddTable2IgorWave("all_Y", Detector->projY, Detector->image_bin, 1)) goto end;
				if (error=AddTable2IgorWave("image", Detector->image, Detector->image_bin, Detector->image_bin)) goto end;
				
				for (i=1, val1=(DOUBLE)Detector->stat[0], val2=0; i<MAXCOINCIDENCEORDER; i++) val2+=(DOUBLE)Detector->stat[i];
				Detector->efficiency=100*val2/(val1+val2);
				if (error = SetIgorFloatingVar("efficiency",&Detector->efficiency,1)) goto end;
	
			break;
	}
	
	// Specific information from the PSD
	switch (Detector->type)  {
		case 0x0F: // CIEL Detector type (X,Y,T) in lines
			error=updateXYDetector(Detector, mode);
			break;
		case 0x17: // Other Detector type (X,Y,T) with delay lines
			break;
		}	
		
end:
  	// return to the original folder
	SetCurrentDataFolder(originalFolder);	
	return error;		
		
}


//-----------------------------------------------------------------------------------------------------------------------------
static short setigorDetector(TDetector *Detector)
{
	DataFolderHandle originalFolder;
	long indices[MAX_DIMENSIONS],dimension[MAX_DIMENSIONS];
	DOUBLE values[2]={0,0};
	short  error=0;

	MemClear(indices, sizeof(indices));
	MemClear(dimension, sizeof(dimension));
	
	if (error = moveigorDetectorFolder(&originalFolder,Detector->num)) goto end;;

  	// Create Igor variables for the detector
  	if (error = SetIgorStringVar("name",Detector->name,1)) goto end;		// Detector name: electron, ion, photon...
  	if (error = SetIgorIntVar("type",Detector->type,1)) goto end;			// Detector type: pixels, delay lines...
  	if (error = SetIgorIntVar("minOrder",Detector->minOrder,1)) goto end;	// Coincidence order to consider : min
  	if (error = SetIgorIntVar("maxOrder",Detector->maxOrder,1)) goto end;	// Coincidence order to consider : max

	if (error = SetIgorIntVar("tof_window",Detector->tof_window,1)) goto end;		// Tof windows	 (micro seconde)
	if (error = SetIgorIntVar("tof_offset",Detector->tof_offset,1)) goto end;		// Tof offset	 (ns seconde)
	if (error = SetIgorIntVar("tof_bin",Detector->tof_bin,1)) goto end;			// bining TDC	 (ps)
	if (error = SetIgorIntVar("image_bin",Detector->image_bin,1)) goto end;			// bining TDC	 (ps)
  	
  	if (error = SetIgorFloatingVar("efficiency",&Detector->efficiency,1)) goto end;		// bining TDC	 (ps)

  	switch(Detector->type) {
  		case 0x0F:	
  				if (error=setigorXYDetector(Detector)) goto end;	
  			break;
  			
  		case 0x17:
  			break;
  	}
  	
end:
  	// return to the original folder
	SetCurrentDataFolder(originalFolder);	
	return error; 
}

//-----------------------------------------------------------------------------------------------------------------------------
static short getigorDetector(TDetector *Detector, int num)
{
	DataFolderHandle originalFolder;
	long indices[MAX_DIMENSIONS],dimension[MAX_DIMENSIONS];
	short error=0;
	DOUBLE val1, val2, values[2]={0,0};
	char name[255];
	
	MemClear(indices, sizeof(indices));
	MemClear(dimension, sizeof(dimension));

	if (error = moveigorDetectorFolder(&originalFolder,num)) goto end;;

  	// Create Igor wave and variables for the list of detector
  	Detector->num=num;
	if (FetchStrVar("name",name)) {  	
  	  	if (error = SetIgorStringVar("name",Detector->name,1)) goto end;
		}
	else strcpy(Detector->name,name);
	strncpy(Detector->suffix,name,1);
  	
   	if (FetchNumVar("type",&val1,&val2)==-1) { 
  		if (error = SetIgorIntVar("type",Detector->type,1)) goto end;
  		}
  	else Detector->type=(long)val1;
  	
    if (FetchNumVar("minOrder",&val1,&val2)==-1) { 
  		if (error = SetIgorIntVar("minOrder",Detector->minOrder,1)) goto end;
  		}
  	else Detector->minOrder=(long)val1;			
    
    if (FetchNumVar("maxOrder",&val1,&val2)==-1) { 
  		if (error = SetIgorIntVar("maxOrder",Detector->maxOrder,1)) goto end;
  		}
  	else Detector->maxOrder=(long)val1;
  	
  	  	// Timing issue 
    if (FetchNumVar("tof_window",&val1,&val2)==-1) { 
  		if (error = SetIgorIntVar("tof_window",Detector->tof_window,1)) goto end;
  		}
  	else Detector->tof_window=(long)val1;

    if (FetchNumVar("tof_offset",&val1,&val2)==-1) { 
  		if (error = SetIgorIntVar("tof_offset",Detector->tof_offset,1)) goto end;
  		}
  	else Detector->tof_offset=(long)val1;
  	
     if (FetchNumVar("tof_bin",&val1,&val2)==-1) { 
  		if (error = SetIgorIntVar("tof_bin",Detector->tof_bin,1)) goto end;
  		}
  	else Detector->tof_bin=(long)val1;
 
      if (FetchNumVar("image_bin",&val1,&val2)==-1) { 
  		if (error = SetIgorIntVar("image_bin",Detector->image_bin,1)) goto end;
  		}
  	else Detector->image_bin=(long)val1;
 
      if (FetchNumVar("efficiency",&val1,&val2)==-1) { 
  		if (error = SetIgorFloatingVar("efficiency",&Detector->efficiency,1)) goto end;
  		}
  	else Detector->efficiency=(long)val1;
  	
  	
  	switch(Detector->type) {
  		case 0x0F:	
  				if (error=getigorXYDetector(Detector)) goto end;	
  			break;
  			
  		case 0x17:
  			break;
  	}

end:
  	// return to the original folder
	SetCurrentDataFolder(originalFolder);	
	return error; 
}
//-----------------------------------------------------------------------------------------------------------------------------
#pragma mark ----------------------------
//-----------------------------------------------------------------------------------------------------------------------------
short createDetectorList(TListDetector *ListOfDetector, int mode)
{
	short error=0;
	
	// Initialization
	ListOfDetector->modeDetector=mode;
	switch (mode) {
		case 2: // mode CIEL 1 : 2 detectors
				ListOfDetector->nbDetector=2;
				ListOfDetector->wordsizeEvent=24;				
				ListOfDetector->maskDetector=0x000040;   // bit 6 <->64
				ListOfDetector->sigbitDetector=sigbit(ListOfDetector->maskDetector);
				ListOfDetector->maskChannel=0x00003F;   
				ListOfDetector->sigbitChannel=sigbit(ListOfDetector->maskChannel);
				ListOfDetector->maskTime=0xFFFF00;   
				ListOfDetector->sigbitTime=sigbit(ListOfDetector->maskTime);
				ListOfDetector->timeBased=0;				
				// Ion detector
				createDetector(&ListOfDetector->Detector[0],0,"ion",0x0F, 1,2,NB_CANAUX_TEMPORELS,0,500,32);				
				createXYDetector(&ListOfDetector->Detector[0], 4,0,6, 0, 1.2e-3, 0,0,0);
				// Electron detector
				createDetector(&ListOfDetector->Detector[1],1,"electron",0x0F,1,2,NB_CANAUX_TEMPORELS,0,500,32);
				createXYDetector(&ListOfDetector->Detector[1], 4,0,3, 0, 1.2e-3, 0,0,0);
				break;
				
		case 3: // mode CIEL 2 : 1 detector
				ListOfDetector->nbDetector=1;
				ListOfDetector->wordsizeEvent=32;
				ListOfDetector->maskDetector=0xC0000000;   // bits 30-31
				ListOfDetector->sigbitDetector=sigbit(ListOfDetector->maskDetector);
				ListOfDetector->maskChannel=0x3FC00000;   
				ListOfDetector->sigbitChannel=sigbit(ListOfDetector->maskChannel);
				ListOfDetector->maskTime=0x003FFFFF;   
				ListOfDetector->sigbitTime=sigbit(ListOfDetector->maskTime);
				ListOfDetector->timeBased=1;
				createDetector(&ListOfDetector->Detector[0],0,"electron",0x0F, 1,2,NB_CANAUX_TEMPORELS,0,100,64);
				createXYDetector(&ListOfDetector->Detector[0], 8,64, 5, 0, 1.2e-3, 2,95,127);								
				break;
	}
setigorDetectorList(ListOfDetector);	
return 0;
}

//-----------------------------------------------------------------------------------------------------------------------------
short setigorDetectorList(TListDetector *ListOfDetector)
{
	DataFolderHandle originalFolder;
	short error=0;
	int i;

	if (error = moveigorFolder(&originalFolder)) goto end;

  	// Create Igor wave and variables for the list of detector
  	if (error = SetIgorIntVar("nbDetector",ListOfDetector->nbDetector,1)) goto end;
  	if (error = SetIgorIntVar("modeDetector",ListOfDetector->modeDetector,1)) goto end;
  	if (error = SetIgorIntVar("wordsizeEvent",ListOfDetector->wordsizeEvent,1)) goto end;
  	if (error = SetIgorIntVar("maskDetector",ListOfDetector->maskDetector,1)) goto end;
  	if (error = SetIgorIntVar("maskChannel",ListOfDetector->maskChannel,1)) goto end;
  	if (error = SetIgorIntVar("maskTime",ListOfDetector->maskTime,1)) goto end;
  	if (error = SetIgorIntVar("timeBased",ListOfDetector->timeBased,1)) goto end;

  	// Handling detectors
  	for (i=0; i<ListOfDetector->nbDetector;i++) {
  		if (error=setigorDetector(&ListOfDetector->Detector[i])) goto end;
  		}
end:
  	// return to the original folder
	SetCurrentDataFolder(originalFolder);
 	return error;
}


//-----------------------------------------------------------------------------------------------------------------------------
short getigorDetectorList(TListDetector *ListOfDetector)
{
	DataFolderHandle originalFolder;
	DOUBLE val1, val2;
	short i, error=0;

	if (error = moveigorFolder(&originalFolder)) goto end;  	

  	// Create Igor wave and variables for the list of detector
  	if (FetchNumVar("modeDetector",&val1,&val2)==-1) { 
  	  	if (error = SetIgorIntVar("modeDetector",ListOfDetector->modeDetector,1)) goto end;
	}
  	else ListOfDetector->modeDetector=(long)val1;
  	
  	if (FetchNumVar("nbDetector",&val1,&val2)==-1) { 
  			if (error = SetIgorIntVar("nbDetector",ListOfDetector->nbDetector,1)) goto end;
  		}
  	else ListOfDetector->nbDetector=(long)val1;

  	
  	// Handling data buffer
  	if (FetchNumVar("wordsizeEvent",&val1,&val2)==-1) { 
  			if (error = SetIgorIntVar("wordsizeEvent",ListOfDetector->wordsizeEvent,1)) goto end;
  		}
  	else ListOfDetector->wordsizeEvent=(long)val1;
  	
  	if (FetchNumVar("maskDetector",&val1,&val2)==-1) { 
  		if (error = SetIgorIntVar("maskDetector",ListOfDetector->maskDetector,1)) goto end;
  		}
  	else ListOfDetector->maskDetector=(long)val1;
  	ListOfDetector->sigbitDetector=sigbit(ListOfDetector->maskDetector);

  	if (FetchNumVar("maskChannel",&val1,&val2)==-1) { 
  		if (error = SetIgorIntVar("maskChannel",ListOfDetector->maskChannel,1)) goto end;
  		}
  	else ListOfDetector->maskChannel=(long)val1;
  	ListOfDetector->sigbitChannel=sigbit(ListOfDetector->maskChannel);

  	if (FetchNumVar("maskTime",&val1,&val2)==-1) { 
  		if (error = SetIgorIntVar("maskTime",ListOfDetector->maskTime,1)) goto end;
  		}
  	else ListOfDetector->maskTime=(long)val1;
  	ListOfDetector->sigbitTime=sigbit(ListOfDetector->maskTime);				

  	if (FetchNumVar("timeBased",&val1,&val2)==-1) { 
  		if (error = SetIgorIntVar("timeBased",ListOfDetector->timeBased,1)) goto end;
  		}
  	else ListOfDetector->timeBased=(long)val1;
  	  	
  	// Handling detectors
  	for (i=0; i<ListOfDetector->nbDetector;i++) {
  		if (error=getigorDetector(&ListOfDetector->Detector[i], i)) goto end;
  		}
 
 end:
  	// return to the original folder
	SetCurrentDataFolder(originalFolder);
 	return error;
 }
 
//-----------------------------------------------------------------------------------------------------------------------------
// All structures are 2-byte-aligned.
#pragma mark ----------------------------

#if TARGET_CPU_PPC
	#pragma options align=reset
#endif
#ifdef _WINDOWS_
	#pragma pack()
#endif

